//
// Tahoe RESTful Web API in JavaScript
// Copyright (C) 2009 Toby Murray
// Released under the terms of the GNU GPL version 2
//
// This library contains code [1] from the Waterken ref_send API which is
// Copyright (C) 2007 Waterken Inc. and can be found at
// http://waterken.sourceforge.net/
//
// [1] In particular, the 'enqueue' code that implements an event-loop, and
//     the 'origin' code which (with 'enqueue') implements asynchronous,
//     in order, HTTP messaging
//
var tahoe = function (http, setTimeout, JSON) {
    var enqueue, origin;
    // Whether or not setTimeout(, 0) executes tasks in order is unspecified, so
    // a FIFO event loop is implemented below.
    enqueue = function () {
        var active = false,
        pending =  [ /* task */ ],
        run = function () {
            var task = pending.shift();
            if (0 === pending.length) {
                active = false;
            } else {
                setTimeout(run, 0);
            }
            task();
        };
        return function (task) {
            pending.push(task);
            if (!active) {
                setTimeout(run, 0);
                active = true;
            }
        };
    }();

    function makePromiseResolverPair() {
        // states of a promise
        var UNRESOLVED = 0,
        FULFILLED = 1,
        BROKEN = -1,
        state = UNRESOLVED, // one of UNRESOLVED, RESOLVED or BROKEN
        // callbacks on fulfilled or broken
        cbFulfilled = null,
        cbBroken = null,
        value = null,
        promise = {
            getValue: function () {
                if (FULFILLED === state) {
                    return value;
                } else if (BROKEN === state) {
                    throw value;
                } else {
                    return promise;
                }
            },
            when: function (f, b) {
                cbFulfilled = f;
                cbBroken = b;
                if (FULFILLED === state) {
                    enqueue(function () {
                        cbFulfilled(value);
                    });
                } else if (BROKEN === state) {
                    enqueue(function () {
                        cbBroken(value);
                    });
                }
            }
        },
        resolver = {
            fulfill: function (v) {
                if (UNRESOLVED !== state) { 
                    throw "fulfilling resolved promise"; 
                }
                value = v;
                state = FULFILLED;
                if (null !== cbFulfilled) {
                    enqueue(function () {
                        cbFulfilled(value);
                    });
                }
            },
            smash: function (reason) {
                if (UNRESOLVED !== state) { 
                    throw "smashing resolved promise"; 
                }
                value = reason;
                state = BROKEN;
                if (null !== cbBroken) {
                    enqueue(function () {
                        cbBroken(value);
                    });
                }
            }
        };
        return [promise, resolver];
    }


    function makeMessage(method, URL, argv, receiveFunc) {
        var pr = makePromiseResolverPair();
        return {
            method: method,
            URL:  URL,
            argv: argv,
            receive: receiveFunc,
            // promise and resolver for the result of this message
            promise: pr[0],
            resolver: pr[1]
        };
    }

    origin = function () {
        var active = false,
        pending = [ /* Message */ ],
        output = function () {
            var m = pending[0];
            http.open(m.method, m.URL, true);
            http.onreadystatechange = function () {
                if (4 !== http.readyState) {
                    return;
                }
                if (m !== pending.shift()) {
                    throw 'problem';
                }
                if (0 === pending.length) {
                    active = false;
                } else {
                    enqueue(output);
                }
                m.receive(http, m);
            };
            http.send(m.argv);
        };

        return {
            send: function (msg) {
                pending.push(msg);
                if (!active) {
                    enqueue(output);
                    active = true;
                }
            }
        };
    }();
    
    function receiveJSON(http, msg) {
        var value;
        // we expect to receive JSON and will resolve msg's promise with
        // the result of parsing the JSON received or smash it otherwise
        if (200 === http.status) {
            // TODO: check Content-Type if Tahoe starts setting this correctly
            try {
                value = JSON.parse(http.responseText, null);
                msg.resolver.fulfill(value);
            } catch (e) {
                msg.resolver.smash(e);
            } 
        } else {
            msg.resolver.smash('unexpected HTTP response: ' + http.status);
        }
    }

    function validateURI(uri) {
        if (uri.indexOf('#') !== -1) { 
            throw "uri contains fragment";
        }
        if (uri.indexOf('?') !== -1) {
            throw "uri contains arguments";
        }
    }


    function stat(uri) {
        var msg, reqUri = '/uri/' + uri;
        
        validateURI(uri);
        reqUri += '?t=json';
        msg = makeMessage('GET', reqUri, null, receiveJSON);
        origin.send(msg);
        return msg.promise;
    }

    function appendChild(uri, child) {
        var reqUri = uri;
        if (uri.charAt(uri.length - 1) !== '/') {
            reqUri += '/';
        }
        reqUri += child;
        return reqUri;
    }

    function receive(http, msg) {
        if (200 === http.status) {
            msg.resolver.fulfill(http.responseText);
        } else {
            msg.resolver.smash('unexpected HTTP response: ' + http.status);
        }
    }


    function unlink(uri, child) {
        var msg, reqUri = '/uri/' + appendChild(uri, child);
        validateURI(uri);
        msg = makeMessage('DELETE', reqUri, null, receive);
        origin.send(msg);
        return msg.promise;
    }

    function mkdir(uri, child) {
        var msg, reqUri = '/uri/' + appendChild(uri, child);
        validateURI(uri);
        reqUri += '?t=mkdir';
        msg = makeMessage('POST', reqUri, null, receive);
        origin.send(msg);
        return msg.promise;
    }

    function attach(uri, child, childCap, replace) {
        var msg, reqUri = '/uri/' + appendChild(uri, child);
        validateURI(uri);
        reqUri += '?t=uri';
        reqUri += '&replace=' + (replace ? 'true' : 'false');
        msg = makeMessage('PUT', reqUri, childCap, receive);
        origin.send(msg);
        return msg.promise;
    }

    function rename(uri, child, newName) {
        var msg, reqUri = '/uri/' + uri;
        validateURI(uri);
        reqUri += '?t=rename';
        reqUri += '&from_name=' + child;
        reqUri += '&to_name=' + newName;
        msg = makeMessage('POST', reqUri, null, receive);
        origin.send(msg);
        return msg.promise;
    }

    // export the public API
    return {
        stat: stat,
        unlink: unlink,
        mkdir: mkdir,
        attach: attach,
        rename: rename,
        makePromiseResolverPair: makePromiseResolverPair
    };
};
        
        
