
class IReadable():
    """I represent a readable object -- either an immutable file, or a
    specific version of a mutable file.
    """

    def is_readonly():
        """Return True if this reference provides mutable access to the given
        file or directory (i.e. if you can modify it), or False if not. Note
        that even if this reference is read-only, someone else may hold a
        read-write reference to it.

        For an IReadable returned by get_best_readable_version(), this will
        always return True, but for instances of subinterfaces such as
        IMutableFileVersion, it may return False."""

    def is_mutable():
        """Return True if this file or directory is mutable (by *somebody*,
        not necessarily you), False if it is is immutable. Note that a file
        might be mutable overall, but your reference to it might be
        read-only. On the other hand, all references to an immutable file
        will be read-only; there are no read-write references to an immutable
        file."""

    def get_storage_index():
        """Return the storage index of the file."""

    def get_size():
        """Return the length (in bytes) of this readable object."""

    def download_to_data():
        """Download all of the file contents. I return a Deferred that fires
        with the contents as a byte string."""

    def read(consumer, offset=0, size=None):
        """Download a portion (possibly all) of the file's contents, making
        them available to the given IConsumer. Return a Deferred that fires
        (with the consumer) when the consumer is unregistered (either because
        the last byte has been given to it, or because the consumer threw an
        exception during write(), possibly because it no longer wants to
        receive data). The portion downloaded will start at 'offset' and
        contain 'size' bytes (or the remainder of the file if size==None).

        The consumer will be used in non-streaming mode: an IPullProducer
        will be attached to it.

        The consumer will not receive data right away: several network trips
        must occur first. The order of events will be::

         consumer.registerProducer(p, streaming)
          (if streaming == False)::
           consumer does p.resumeProducing()
            consumer.write(data)
           consumer does p.resumeProducing()
            consumer.write(data).. (repeat until all data is written)
         consumer.unregisterProducer()
         deferred.callback(consumer)

        If a download error occurs, or an exception is raised by
        consumer.registerProducer() or consumer.write(), I will call
        consumer.unregisterProducer() and then deliver the exception via
        deferred.errback(). To cancel the download, the consumer should call
        p.stopProducing(), which will result in an exception being delivered
        via deferred.errback().

        See src/allmydata/util/consumer.py for an example of a simple
        download-to-memory consumer.
        """


class IMutableFileVersion(IReadable):
    """I provide access to a particular version of a mutable file. The
    access is read/write if I was obtained from a filenode derived from
    a write cap, or read-only if the filenode was derived from a read cap.
    """

    def get_sequence_number():
        """Return the sequence number of this version."""

    def get_servermap():
        """Return the IMutableFileServerMap instance that was used to create
        this object.
        """

    def get_writekey():
        """Return this filenode's writekey, or None if the node does not have
        write-capability. This may be used to assist with data structures
        that need to make certain data available only to writers, such as the
        read-write child caps in dirnodes. The recommended process is to have
        reader-visible data be submitted to the filenode in the clear (where
        it will be encrypted by the filenode using the readkey), but encrypt
        writer-visible data using this writekey.
        """

    def replace(new_contents):
        """Replace the contents of the mutable file, provided that no other
        node has published (or is attempting to publish, concurrently) a
        newer version of the file than this one.

        I will avoid modifying any share that is different than the version
        given by get_sequence_number(). However, if another node is writing
        to the file at the same time as me, I may manage to update some shares
        while they update others. If I see any evidence of this, I will signal
        UncoordinatedWriteError, and the file will be left in an inconsistent
        state (possibly the version you provided, possibly the old version,
        possibly somebody else's version, and possibly a mix of shares from
        all of these).

        The recommended response to UncoordinatedWriteError is to either
        return it to the caller (since they failed to coordinate their
        writes), or to attempt some sort of recovery. It may be sufficient to
        wait a random interval (with exponential backoff) and repeat your
        operation. If I do not signal UncoordinatedWriteError, then I was
        able to write the new version without incident.

        I return a Deferred that fires (with a PublishStatus object) when the
        update has completed.
        """

    def modify(modifier_cb):
        """Modify the contents of the file, by downloading this version,
        applying the modifier function (or bound method), then uploading
        the new version. This will succeed as long as no other node
        publishes a version between the download and the upload.
        I return a Deferred that fires (with a PublishStatus object) when
        the update is complete.

        The modifier callable will be given three arguments: a string (with
        the old contents), a 'first_time' boolean, and a servermap. As with
        download_to_data(), the old contents will be from this version,
        but the modifier can use the servermap to make other decisions
        (such as refusing to apply the delta if there are multiple parallel
        versions, or if there is evidence of a newer unrecoverable version).
        'first_time' will be True the first time the modifier is called,
        and False on any subsequent calls.

        The callable should return a string with the new contents. The
        callable must be prepared to be called multiple times, and must
        examine the input string to see if the change that it wants to make
        is already present in the old version. If it does not need to make
        any changes, it can either return None, or return its input string.

        If the modifier raises an exception, it will be returned in the
        errback.
        """


# The hierarchy looks like this:
#  IFilesystemNode
#   IFileNode
#    IMutableFileNode
#    IImmutableFileNode
#   IDirectoryNode

class IFilesystemNode(Interface):
    def get_cap():
        """Return the strongest 'cap instance' associated with this node.
        (writecap for writeable-mutable files/directories, readcap for
        immutable or readonly-mutable files/directories). To convert this
        into a string, call .to_string() on the result."""

    def get_readcap():
        """Return a readonly cap instance for this node. For immutable or
        readonly nodes, get_cap() and get_readcap() return the same thing."""

    def get_repair_cap():
        """Return an IURI instance that can be used to repair the file, or
        None if this node cannot be repaired (either because it is not
        distributed, like a LIT file, or because the node does not represent
        sufficient authority to create a repair-cap, like a read-only RSA
        mutable file node [which cannot create the correct write-enablers]).
        """

    def get_verify_cap():
        """Return an IVerifierURI instance that represents the
        'verifiy/refresh capability' for this node. The holder of this
        capability will be able to renew the lease for this node, protecting
        it from garbage-collection. They will also be able to ask a server if
        it holds a share for the file or directory.
        """

    def get_uri():
        """Return the URI string corresponding to the strongest cap associated
        with this node. If this node is read-only, the URI will only offer
        read-only access. If this node is read-write, the URI will offer
        read-write access.

        If you have read-write access to a node and wish to share merely
        read-only access with others, use get_readonly_uri().
        """

    def get_write_uri(n):
        """Return the URI string that can be used by others to get write
        access to this node, if it is writeable. If this is a read-only node,
        return None."""

    def get_readonly_uri():
        """Return the URI string that can be used by others to get read-only
        access to this node. The result is a read-only URI, regardless of
        whether this node is read-only or read-write.

        If you have merely read-only access to this node, get_readonly_uri()
        will return the same thing as get_uri().
        """

    def get_storage_index():
        """Return a string with the (binary) storage index in use on this
        download. This may be None if there is no storage index (i.e. LIT
        files)."""

    def is_readonly():
        """Return True if this reference provides mutable access to the given
        file or directory (i.e. if you can modify it), or False if not. Note
        that even if this reference is read-only, someone else may hold a
        read-write reference to it."""

    def is_mutable():
        """Return True if this file or directory is mutable (by *somebody*,
        not necessarily you), False if it is is immutable. Note that a file
        might be mutable overall, but your reference to it might be
        read-only. On the other hand, all references to an immutable file
        will be read-only; there are no read-write references to an immutable
        file.
        """

    def is_unknown():
        """Return True if this is an unknown node."""

    def is_allowed_in_immutable_directory():
        """Return True if this node is allowed as a child of a deep-immutable
        directory. This is true if either the node is of a known-immutable type,
        or it is unknown and read-only.
        """

    def raise_error():
        """Raise any error associated with this node."""

    def get_size():
        """Return the length (in bytes) of the data this node represents. For
        directory nodes, I return the size of the backing store. I return
        synchronously and do not consult the network, so for mutable objects,
        I will return the most recently observed size for the object, or None
        if I don't remember a size. Use get_current_size, which returns a
        Deferred, if you want more up-to-date information."""

    def get_current_size():
        """I return a Deferred that fires with the length (in bytes) of the
        data this node represents.
        """


class IFileNode(IFilesystemNode):
    """I am a node representing a file: a sequence of bytes. I am not a
    container, like IDirectoryNode."""

    def get_best_readable_version():
        """Return a Deferred that fires with an IReadable for the 'best'
        available version of the file. The IReadable provides only read
        access, even if this filenode was derived from a write cap.

        For an immutable file, there is only one version. For a mutable
        file, the 'best' version is the recoverable version with the
        highest sequence number. If no uncoordinated writes have occurred,
        and if enough shares are available, then this will be the most
        recent version that has been uploaded. If no version is recoverable,
        the Deferred will errback with an UnrecoverableFileError.
        """

    def download_best_version():
        """Download the contents of the version that would be returned
        by get_best_readable_version(). This is equivalent to calling
        download_to_data() on the IReadable given by that method.

        I return a Deferred that fires with a byte string when the file
        has been fully downloaded. To support streaming download, use
        the 'read' method of IReadable. If no version is recoverable,
        the Deferred will errback with an UnrecoverableFileError.
        """

    def get_size_of_best_version():
        """Find the size of the version that would be returned by
        get_best_readable_version().

        I return a Deferred that fires with an integer. If no version
        is recoverable, the Deferred will errback with an
        UnrecoverableFileError.
        """


class IImmutableFileNode(IFileNode, IReadable):
    """I am a node representing an immutable file. Immutable files have
    only one version."""


class IMutableFileNode(IFileNode):
    """I provide access to a 'mutable file', which retains its identity
    regardless of what contents are put in it.

    The consistency-vs-availability problem means that there might be
    multiple versions of a file present in the grid, some of which might be
    unrecoverable (i.e. have fewer than 'k' shares). These versions are
    loosely ordered: each has a sequence number and a hash, and any version
    with seqnum=N was uploaded by a node which has seen at least one version
    with seqnum=N-1.

    The 'servermap' (an instance of IMutableFileServerMap) is used to
    describe the versions that are known to be present in the grid, and which
    servers are hosting their shares. It is used to represent the 'state of
    the world', and is used for this purpose by my test-and-set operations.
    Downloading the contents of the mutable file will also return a
    servermap. Uploading a new version into the mutable file requires a
    servermap as input, and the semantics of the replace operation is
    'replace the file with my new version if it looks like nobody else has
    changed the file since my previous download'. Because the file is
    distributed, this is not a perfect test-and-set operation, but it will do
    its best. If the replace process sees evidence of a simultaneous write,
    it will signal an UncoordinatedWriteError, so that the caller can take
    corrective action.


    Most readers will want to use the 'best' current version of the file,
    and should use my 'get_best_mutable_version()' method (or
    'get_best_readable_version()' for read-only access).

    To unconditionally replace the file, callers should use overwrite(). This
    is the mode that user-visible mutable files will probably use.

    To apply some delta to the file, call modify() with a callable modifier
    function that can apply the modification that you want to make. This is
    the mode that dirnodes will use, since most directory modification
    operations can be expressed in terms of deltas to the directory state.


    Three methods are available for users who need to perform more complex
    operations. The first is get_servermap(), which returns an up-to-date
    servermap using a specified mode. The second is download_version(), which
    downloads a specific version (not necessarily the 'best' one). The third
    is 'upload', which accepts new contents and a servermap (which must have
    been updated with MODE_WRITE). The upload method will attempt to apply
    the new contents as long as no other node has modified the file since the
    servermap was updated. This might be useful to a caller who wants to
    merge multiple versions into a single new one.

    Note that each time the servermap is updated, a specific 'mode' is used,
    which determines how many peers are queried. To use a servermap for my
    replace() method, that servermap must have been updated in MODE_WRITE.
    These modes are defined in allmydata.mutable.common, and consist of
    MODE_READ, MODE_WRITE, MODE_ANYTHING, and MODE_CHECK. Please look in
    allmydata/mutable/servermap.py for details about the differences.

    Mutable files are currently limited in size (about 3.5MB max) and can
    only be retrieved and updated all-at-once, as a single big string. Future
    versions of our mutable files will remove this restriction.
    """

    def get_best_mutable_version():
        """Return a Deferred that fires with an IMutableFileVersion for
        the 'best' available version of the file. The best version is
        the recoverable version with the highest sequence number. If no
        uncoordinated writes have occurred, and if enough shares are
        available, then this will be the most recent version that has
        been uploaded.

        If no version is recoverable, the Deferred will errback with an
        UnrecoverableFileError.
        """

    def overwrite(new_contents):
        """Unconditionally replace the contents of the mutable file with new
        ones. This simply chains get_servermap(MODE_WRITE) and upload(). This
        is only appropriate to use when the new contents of the file are
        completely unrelated to the old ones, and you do not care about other
        clients' changes.

        I return a Deferred that fires (with a PublishStatus object) when the
        update has completed.
        """

    def modify(modifier_cb):
        """Modify the contents of the file, by downloading the current
        version, applying the modifier function (or bound method), then
        uploading the new version. I return a Deferred that fires (with a
        PublishStatus object) when the update is complete.

        The modifier callable will be given three arguments: a string (with
        the old contents), a 'first_time' boolean, and a servermap. As with
        download_best_version(), the old contents will be from the best
        recoverable version, but the modifier can use the servermap to make
        other decisions (such as refusing to apply the delta if there are
        multiple parallel versions, or if there is evidence of a newer
        unrecoverable version). 'first_time' will be True the first time the
        modifier is called, and False on any subsequent calls.

        The callable should return a string with the new contents. The
        callable must be prepared to be called multiple times, and must
        examine the input string to see if the change that it wants to make
        is already present in the old version. If it does not need to make
        any changes, it can either return None, or return its input string.

        If the modifier raises an exception, it will be returned in the
        errback.
        """

    def get_servermap(mode):
        """Return a Deferred that fires with an IMutableFileServerMap
        instance, updated using the given mode.
        """

    def download_version(servermap, version):
        """Download a specific version of the file, using the servermap
        as a guide to where the shares are located.

        I return a Deferred that fires with the requested contents, or
        errbacks with UnrecoverableFileError. Note that a servermap which was
        updated with MODE_ANYTHING or MODE_READ may not know about shares for
        all versions (those modes stop querying servers as soon as they can
        fulfil their goals), so you may want to use MODE_CHECK (which checks
        everything) to get increased visibility.
        """

    def upload(new_contents, servermap):
        """Replace the contents of the file with new ones. This requires a
        servermap that was previously updated with MODE_WRITE.

        I attempt to provide test-and-set semantics, in that I will avoid
        modifying any share that is different than the version I saw in the
        servermap. However, if another node is writing to the file at the
        same time as me, I may manage to update some shares while they update
        others. If I see any evidence of this, I will signal
        UncoordinatedWriteError, and the file will be left in an inconsistent
        state (possibly the version you provided, possibly the old version,
        possibly somebody else's version, and possibly a mix of shares from
        all of these).

        The recommended response to UncoordinatedWriteError is to either
        return it to the caller (since they failed to coordinate their
        writes), or to attempt some sort of recovery. It may be sufficient to
        wait a random interval (with exponential backoff) and repeat your
        operation. If I do not signal UncoordinatedWriteError, then I was
        able to write the new version without incident.

        I return a Deferred that fires (with a PublishStatus object) when the
        publish has completed. I will update the servermap in-place with the
        location of all new shares.
        """

    def get_writekey():
        """Return this filenode's writekey, or None if the node does not have
        write-capability. This may be used to assist with data structures
        that need to make certain data available only to writers, such as the
        read-write child caps in dirnodes. The recommended process is to have
        reader-visible data be submitted to the filenode in the clear (where
        it will be encrypted by the filenode using the readkey), but encrypt
        writer-visible data using this writekey.
        """
